<?php

namespace App\Http\Controllers;

use App\Models\Logo;
use Illuminate\Http\Request;

class LogoController extends Controller
{


public function index()
{
    // الحصول على أحدث شعار من قاعدة البيانات
    $latestLogo = Logo::latest()->first();

    if ($latestLogo) {
        // تكوين رابط الصورة باستخدام وظيفة asset
        $imgUrl = asset('images/' . $latestLogo->img);

        // إرجاع استجابة JSON تحتوي على عنوان الشعار ورابط الصورة
        return response()->json([
            'title' => $latestLogo->title, // افترض أن عمود العنوان موجود في قاعدة البيانات
             'color' => $latestLogo->color,
            
            'img_url' => $imgUrl,
        ]);
    } else {
        // في حال عدم وجود أي شعارات، إرجاع رسالة خطأ 404
        return response()->json(['message' => 'No logos found.'], 404);
    }
}


public function store(Request $request)
{
    $request->validate([
        'title' => 'required|string|max:255',
        'img' => 'nullable|image',
        'color' => 'nullable|string|max:20', // إضافة التحقق للحقل الجديد
    ]);

    if ($request->hasFile('img')) {
        $image = $request->file('img');
        $imageName = time() . '.' . $image->getClientOriginalExtension();
        $image->move(public_path('images'), $imageName);
    } else {
        $imageName = null;
    }

    $logo = Logo::create([
        'title' => $request->title,
        'img' => $imageName,
        'color' => $request->color, // تخزين قيمة اللون
    ]);

    return response()->json([
        'title' => $logo->title,
        'img_url' => $imageName ? asset('images/' . $imageName) : null,
        'color' => $logo->color, // إضافة اللون في الاستجابة
    ], 201);
}


    public function show($id)
    {
        $logo = Logo::findOrFail($id);

        return response()->json([
            'img_url' => asset('images/' . $logo->img),
        ]);
    }

public function update(Request $request, $id)
{
    // Validate incoming requests
$validatedData = $request->validate([
    'title' => 'required|string|max:255',
    'img' => 'nullable|image|mimes:jpg,jpeg,png,bmp|max:2048',
    'color' => 'nullable|string|max:20', // إضافة التحقق للحقل الجديد
]);




    try {
        // Find the logo by ID
        $logo = Logo::findOrFail($id);

        // Update the logo with the request data
        $logo->title = $validatedData['title'];
$logo->color = $validatedData['color']; 
        if ($request->hasFile('img')) {
            // حذف الصورة القديمة إذا كانت موجودة
            if ($logo->img) {
                $existingImagePath = public_path('images/' . $logo->img);
                if (file_exists($existingImagePath)) {
                    unlink($existingImagePath);
                }
            }

            // حفظ الصورة الجديدة
            $image = $request->file('img');
            $imageName = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('images'), $imageName);

            $logo->img = $imageName;
        }

        $logo->save();

        // Return the updated logo as JSON response with status 200
        return response()->json([
            'title' => $logo->title,
            'img_url' => $logo->img ? asset('images/' . $logo->img) : null,
        ], 200);

    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        // Handle case where the logo was not found
        return response()->json(['error' => 'Logo not found.'], 404);

    } catch (\Illuminate\Validation\ValidationException $e) {
        // Handle validation exceptions
        return response()->json(['error' => 'Validation failed: ' . $e->getMessage()], 422);

    } catch (\Exception $e) {
        // Handle any other exceptions
        return response()->json(['error' => 'Update failed: ' . $e->getMessage()], 500);
    }
}


    public function destroy($id)
    {
        try {
            // Find the logo by ID
            $logo = Logo::findOrFail($id);
            
            // Delete the logo image file from the server
            if ($logo->img) {
                unlink(public_path('images/' . $logo->img));
            }

            // Delete the logo record from the database
            $logo->delete();

            return response()->json(['message' => 'Logo deleted successfully.'], 200);

        } catch (\Exception $e) {
            // Handle exceptions if the logo is not found or any other errors
            return response()->json(['error' => 'Logo not found or delete failed.'], 404);
        }
    }
}
