<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Hash;



class AuthController extends Controller
{
    /**
     * Create a new AuthController instance.
     *
     * @return void
     */
    public function __construct() {
     $this->middleware('auth:api', ['except' => ['login', 'register', 'adminLogin', 'adminRegister']]);
    }

    /**
     * Get a JWT via given credentials.
     *
     * @return \Illuminate\Http\JsonResponse
     */
  
  
  public function getAllUsers()
{
    // التحقق من كون المستخدم مُسجل الدخول
    $authUser = auth()->user();

    // التحقق من أن المستخدم مسؤول (admin)
    if ($authUser->role !== 'admin') {
        return response()->json(['error' => 'Unauthorized'], 401);
    }

    // استرجاع جميع المستخدمين
    $users = User::all();

    // إرجاع قائمة المستخدمين كاستجابة JSON
    return response()->json([
        'message' => 'All registered users retrieved successfully',
        'users' => $users,
    ]);
}

  
  

public function login(Request $request) {
    $validator = Validator::make($request->all(), [
        'phone' => 'required|string',
    ]);

    if ($validator->fails()) {
        return response()->json($validator->errors(), 422);
    }

    $credentials = $request->only('phone');

    // ابحث عن المستخدم باستخدام رقم الهاتف
    $user = User::where('phone', $credentials['phone'])->first();

    if (!$user) {
        return response()->json(['error' => 'User not found'], 404);
    }

    // إنشاء التوكن JWT
    $token = JWTAuth::fromUser($user);

    // إرجاع التوكن ومعلومات المستخدم
    return response()->json([
        'token' => $token,
        'user' => [
            'id' => $user->id, // إضافة المعرف
            'name' => $user->name,
            'email' => $user->email,
            'phone' => $user->phone,
        ]
    ]);
}


 public function adminRegister(Request $request) {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|between:2,100',
            'email' => 'required|string|email|max:100|unique:users',
            'phone' => 'nullable|string|max:15|unique:users',
            'password' => 'required|string|min:6',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors()->toJson(), 400);
        }

        $admin = User::create(array_merge(
            $validator->validated(),
            ['password' => bcrypt($request->password)],
          
            ['role' => 'admin']
        ));

        return response()->json([
            'message' => 'Admin successfully registered',
            'admin' => $admin
        ], 201);
    }

   public function adminLogin(Request $request) {
    $validator = Validator::make($request->all(), [
        'email' => 'required|string|email|max:100',
        'password' => 'required|string|min:6',
    ]);

    if ($validator->fails()) {
        return response()->json($validator->errors(), 422);
    }

    $credentials = $request->only('email', 'password');

    // Find the admin user by email and role
    $admin = User::where('email', $credentials['email'])->where('role', 'admin')->first();

    if (! $admin || ! Hash::check($credentials['password'], $admin->password)) {
        return response()->json(['error' => 'Unauthorized'], 401);
    }

    // Generate JWT token for the admin
    $token = JWTAuth::fromUser($admin);

    // Return token and admin information
    return response()->json([
        'token' => $token,
        'admin' => [
            'name' => $admin->name,
            'email' => $admin->email,
            
        ]
    ]);
}

    /**
     * Register a User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request) {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|between:2,100',
            'email' => 'required|string|email|max:100|unique:users',
            'phone' => 'required|string|max:15|unique:users',
            'password' => 'nullable|string|min:6',
        ]);

        if($validator->fails()){
            return response()->json($validator->errors()->toJson(), 400);
        }

        $user = User::create(array_merge(
                    $validator->validated(),
                    ['password' => bcrypt($request->password)]
                ));

        return response()->json([
            'message' => 'User successfully registered',
            'user' => $user
        ], 201);
    }


    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout() {
        auth()->logout();

        return response()->json(['message' => 'User successfully signed out']);
    }

    /**
     * Refresh a token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
   // public function refresh() {
     //   return $this->createNewToken(auth()->refresh());
    //}
    public function refresh() {
        try {
            $newToken = JWTAuth::refresh(JWTAuth::getToken());
            return $this->createNewToken($newToken);
        } catch (\Tymon\JWTAuth\Exceptions\JWTException $e) {
            return response()->json(['error' => 'Could not refresh token'], 401);
        }
    }



    /**
     * Get the authenticated User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
   
    public function userProfile()
    {
        // تأكد من أن المستخدم مُصادق
        $user = auth()->user();

        // تحقق إذا كان المستخدم مديرًا
        if ($user->role === 'admin') {
            // استرجع جميع المستخدمين
            $users = User::all();

            // أرجع جميع المستخدمين كاستجابة JSON
            return response()->json($users);
        }

        // أرجع بيانات المستخدم إذا لم يكن مديرًا
        return response()->json($user);
   
    }



public function deleteUserByName(Request $request)
{
    // Ensure the user is authenticated and authorized to delete
    $user = auth()->user();

    // Check if the authenticated user is admin
    if ($user->role !== 'admin') {
        return response()->json(['error' => 'Unauthorized'], 401);
    }

    // Validate the request data
    $validator = Validator::make($request->all(), [
        'name' => 'required|string', // Assuming 'name' is the field to identify the user
    ]);

    if ($validator->fails()) {
        return response()->json($validator->errors(), 422);
    }

    $name = $request->input('name');

    // Find the user by name
    $userToDelete = User::where('name', $name)->first();

    // If user not found
    if (!$userToDelete) {
        return response()->json(['error' => 'User not found'], 404);
    }

    try {
        // Delete the user
        $userToDelete->delete();

        return response()->json(['message' => 'User deleted successfully']);
    } catch (\Exception $e) {
        // Handle any errors that occur during deletion
        return response()->json(['error' => 'Failed to delete user'], 500);
    }
}



                       //     تعديل دور المستخدم 

public function changeUserRole(Request $request)
{
    // Ensure the user is authenticated and authorized to change roles
    $authUser = auth()->user();

    // Check if the authenticated user is admin
    if ($authUser->role !== 'admin') {
        return response()->json(['error' => 'Unauthorized'], 401);
    }

    // Validate the request data
    $validator = Validator::make($request->all(), [
        'email' => 'required|email|exists:users,email', // Ensure email exists
        'role' => 'required|string|in:user,admin,manager', // Only allow valid roles
    ]);

    if ($validator->fails()) {
        return response()->json($validator->errors(), 422);
    }

    // Retrieve the user by email and the new role
    $user = User::where('email', $request->input('email'))->first();
    $newRole = $request->input('role');

    // Update the user's role
    $user->role = $newRole;
    $user->save();

    return response()->json([
        'message' => 'User role updated successfully',
        'user' => [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'role' => $user->role,
        ],
    ]);
}


  

    /**
     * Get the token array structure.
     *
     * @param  string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */
 #   protected function createNewToken($token){
  #      return response()->json([
   #         'access_token' => $token,
    #        'token_type' => 'bearer',
     ##       'expires_in' => auth()->factory()->getTTL() * 60,
       ##]);
    #}
    protected function createNewToken($token){
        $payload = JWTAuth::setToken($token)->getPayload();
        return response()->json([
            'access_token' => $token,
            'token_type' => 'bearer',
            'expires_in' => $payload['exp'] - $payload['iat'],
            'user' => auth()->user()
        ]);
    }

}
