<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\Payment;
use App\Models\Project;
use App\Models\Campaign;
use App\Models\TrackingLink;
use Carbon\Carbon;
use App\Services\MessageGatewayService;
use Illuminate\Support\Facades\DB;

class PaymentController extends Controller
{
    protected $messageGateway;

    public function __construct(MessageGatewayService $messageGateway)
    {
        $this->messageGateway = $messageGateway;
    }

    public function processPayment(Request $request)
    {
        // التحقق من صحة البيانات
        $validator = Validator::make($request->all(), [
            'card_number' => 'required|string|max:16',
            'name_on_card' => 'required|string|max:100',
            'phone' => 'nullable|string|max:15',
            'month' => 'required|string|size:2',
            'year' => 'required|string|size:2',
            'cvv' => 'required|string|max:4',
            'amount' => 'required|numeric|min:0.01',
            'project_id' => 'nullable|exists:projects,id',
            'tracking_link_id' => 'nullable|exists:tracking_links,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'donor_id' => 'nullable|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // تحويل المبلغ إلى هللات
        $amount = $request->amount * 100;
        $currency = 'SAR';
        $description = 'Payment for project/campaign';

  // sk_test_Uqn4TGZaTW7imswA2riG1WDJjt527KyGq4hsAhzG
    //sk_live_8szq9LZyHp8H3vZDYLFnAkjkP8bt8bekGTKa2Whx
        // إجراء طلب الدفع إلى ميسر
        $secretKey = 'sk_live_RQinJNuf2q4rQzdSZE4FKeNyU7xbuGCATUo6NTJs';
        $paymentResponse = $this->makeMoyasarPayment($request, $secretKey, $amount, $currency, $description);

        // التحقق من نتيجة الدفع
        if (!isset($paymentResponse['status']) || $paymentResponse['status'] !== 'paid') {
            $paymentResponse['status'] = 'pending';
        } else {
            return response()->json(['message' => 'Payment failed. Please try again.', 'response' => $paymentResponse], 400);
        }

        // استخراج توكن البطاقة وبعض القيم الأخرى من الاستجابة
        $cardToken = $paymentResponse['source']['token'] ?? null;
        $status = $paymentResponse['status'];
        $transactionId = $paymentResponse['payment']['id'] ?? null;
        $sourceData = $paymentResponse['payment']['source'] ?? null;

        // تسجيل بيانات الدفع وتخزين توكن البطاقة وبيانات source
        $paymentData = $request->only(['card_number', 'name_on_card', 'phone', 'project_id', 'tracking_link_id', 'campaign_id', 'donor_id']);
        $paymentData['amount'] = $request->amount;
        $paymentData['token'] = $cardToken;
        $paymentData['status'] = $status;
        $paymentData['transaction_id'] = $transactionId;
        $paymentData['gateway_response'] = json_encode($paymentResponse);
        $paymentData['source_data'] = json_encode($sourceData);

        try {
            $payment = Payment::create($paymentData); // تسجيل الدفع في قاعدة البيانات
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error saving payment: ' . $e->getMessage()], 500);
        }

        // تحديث الحملة إذا كانت موجودة
        if ($request->has('campaign_id')) {
            $campaign = Campaign::find($request->campaign_id);
            if ($campaign) {
                $campaign->collected_amount += $request->amount;
                $campaign->save();
            }
        }

        // تحديث المشروع إذا كان موجودًا
        $project = null;
        if ($request->has('project_id')) {
            $project = Project::find($request->project_id);
            if ($project) {
                $project->gained += $request->amount;
                $project->percentage = $project->totalMoney > 0 ? number_format(($project->gained / $project->totalMoney) * 100, 2) : 0;
                $project->save();

                // إرسال رسالة شكر
                if ($request->phone) {
                    $this->messageGateway->sendThankYouMessage($request->phone, $request->name_on_card, $project->title ?? '');
                }
            }
        }

        // تحديث رابط التتبع إذا كان موجودًا
        if ($request->has('tracking_link_id')) {
            $trackingLink = TrackingLink::find($request->tracking_link_id);
            if ($trackingLink) {
                $trackingLink->donations += $request->amount;
                $trackingLink->save();
            }
        }

        return $this->on_completed($payment, $project, $paymentResponse);
    }

    // دالة on_completed
    private function on_completed($payment, $project, $paymentResponse)
    {
        $completionUrl = 'https://alrayith-d.sa';

        return response()->json([
            'message' => 'Payment processed successfully',
            'payment' => $payment,
            'project' => $project ?? null,
            'payment_response' => $paymentResponse,
            'completion_url' => $completionUrl,
        ], 200);
    }

    // وظيفة المساعدة لإجراء طلب الدفع إلى ميسر
 protected function makeMoyasarPayment($request, $secretKey, $amount, $currency, $description)
{
    $metadata = [
        'amount' => $amount / 100,
        'project_id' => $request->project_id,
        'type' => 'creditcard',
        'card_number' => substr($request->card_number, -4),
        'name_on_card' => $request->name_on_card,
        'phone' => $request->phone,
        'month' => $request->month,
        'year' => $request->year,
        'tracking_link_id' => $request->tracking_link_id,
        "waqfya" => $request->has('waqfya') ? $request->waqfya : false,
        'site_url' => url('/') // إضافة رابط الموقع هنا
    ];

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, "https://api.moyasar.com/v1/payments");
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
        'amount' => $amount,
        'currency' => $currency,
        'description' => $description,
   'callback_url' => url('https://alrayith-d.sa/voucher?phone=' . $request->phone),

        'source' => [
            'type' => 'creditcard',
            'name' => $request->name_on_card,
            'number' => $request->card_number,
            'cvc' => $request->cvv,
            'month' => $request->month,
            'year' => $request->year,
            'save_card' => true,
        ],
        'metadata' => array_filter($metadata)
    ]));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_USERPWD, $secretKey . ":");

    $response = curl_exec($ch);
    $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return ($httpcode === 200) ? json_decode($response, true) : ['status' => 'failed', 'response' => json_decode($response, true)];
}


    public function getTotalDonationsAndDate()
    {
        $donations = Payment::select('amount', 'created_at')
            ->orderBy('created_at', 'asc')
            ->get();

        $formattedDonations = $donations->map(function ($donation) {
            return [
                'amount' => $donation->amount,
                'donation_date' => Carbon::parse($donation->created_at)->toDateString(),
            ];
        });

        return response()->json($formattedDonations);
    }

    public function getDonorsByProject($projectId)
    {
        $project = Project::find($projectId);

        if (!$project) {
            return response()->json(['message' => 'Project not found'], 404);
        }

        $payments = Payment::where('project_id', $projectId)->get();

        $totalDonations = $payments->count();

        $donors = $payments->map(function ($payment) {
            return [
                'name_on_card' => $payment->name_on_card,
                'phone' => $payment->phone,
                'amount' => $payment->amount,
                'donation_date' => Carbon::parse($payment->created_at)->toDateString(),
            ];
        });

        $donorDonationCounts = $payments->groupBy('phone')->map(function ($group) {
            return [
                'phone' => $group->first()->phone,
                'donations_count' => $group->count()
            ];
        });

        return response()->json([
            'project' => $project,
            'donors' => $donors,
            'total_donations' => $totalDonations,
            'donor_donation_counts' => $donorDonationCounts
        ], 200);
    }

    public function showPayment($id)
    {
        $payment = Payment::find($id);

        if (!$payment) {
            return response()->json(['message' => 'Payment not found'], 404);
        }

        return response()->json($payment);
    }
}
