<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Log;
use App\Models\TrackingLink;
use App\Models\Coordinator;
use Illuminate\Http\Request;

class TrackingLinkController extends Controller
{
    public function create(Request $request)
    {
        $messages = [
            'title.required' => 'حقل العنوان مطلوب.',
            'title.string' => 'يجب أن يكون العنوان نصًا.',
            'title.max' => 'يجب ألا يتجاوز العنوان 255 حرفًا.',
            'coordinator_id.required' => 'حقل معرف المنسق مطلوب.',
            'coordinator_id.exists' => 'معرف المنسق غير موجود.',
            'custom_keyword.string' => 'يجب أن تكون الكلمة الدلالية نصًا.',
            'custom_keyword.max' => 'يجب ألا تتجاوز الكلمة الدلالية 255 حرفًا.',
            'added_date.required' => 'حقل تاريخ الإضافة مطلوب.',
            'added_date.date' => 'يجب أن يكون تاريخ الإضافة تاريخًا صحيحًا.',
        ];

        $validatedData = $request->validate([
            'title' => 'required|string|max:255',
            'coordinator_id' => 'required|exists:coordinators,id',
            'custom_keyword' => 'nullable|string|max:255',
            'added_date' => 'required|date',
        ], $messages);

        $coordinator = Coordinator::findOrFail($validatedData['coordinator_id']);

        // إنشاء الرابط التتبعي الجديد
        $trackingUrl = "https://alastura.com?coordinator_id={$coordinator->id}&custom_keyword=" . urlencode($validatedData['custom_keyword'] ?? '');

        $trackingLink = TrackingLink::create([
            'title' => $validatedData['title'],
            'custom_keyword' => $validatedData['custom_keyword'] ?? '',
            'added_date' => $validatedData['added_date'],
            'url' => $trackingUrl,
            'coordinator_id' => $coordinator->id,
            'advertiser_name' => $coordinator->name,
            'advertiser_phone' => $coordinator->phone,
            'advertiser_email' => $coordinator->email,
        ]);

        Log::info('تم إنشاء رابط تتبع جديد:', ['tracking_link_id' => $trackingLink->id, 'title' => $trackingLink->title]);

        $trackingLinkUrl = url("api/tracking/{$trackingLink->id}");

        return response()->json(['tracking_link' => $trackingLinkUrl]);
    }

    public function track($id, Request $request)
    {
        $trackingLink = TrackingLink::find($id);

        if (!$trackingLink) {
            Log::error('رابط التتبع غير موجود', ['id' => $id]);
            return response()->json(['message' => 'رابط التتبع غير موجود'], 404);
        }

        // زيادة عدد الزيارات
        $trackingLink->increment('visits');
        Log::info('تم زيادة عدد الزيارات', ['tracking_link_id' => $id, 'visits' => $trackingLink->visits]);

        // إذا كان هناك عملية شراء
        if ($request->input('purchase') === 'true') {
            $trackingLink->increment('purchases_count');
            Log::info('تم زيادة عدد عمليات الشراء', ['tracking_link_id' => $id, 'purchases_count' => $trackingLink->purchases_count]);
        }

        // إذا كان هناك معلومات عن التبرع
        if ($request->input('donation_id')) {
            $donationId = $request->input('donation_id');
            $donationAmount = $request->input('donation_amount', 0);

            $trackingLink->increment('donations');
            $trackingLink->update([
                'last_donation_id' => $donationId,
                'last_donation_amount' => $donationAmount,
            ]);

            Log::info('تم تحديث تفاصيل التبرع', [
                'tracking_link_id' => $id,
                'donation_id' => $donationId,
                'donation_amount' => $donationAmount,
                'donations' => $trackingLink->donations,
            ]);
        }

        // تأكد من صحة الرابط قبل التوجيه
        $redirectUrl = $trackingLink->url;

        if (!filter_var($redirectUrl, FILTER_VALIDATE_URL)) {
            Log::error('الرابط غير صحيح', ['tracking_link_id' => $id, 'url' => $redirectUrl]);
            return response()->json(['message' => 'الرابط غير صحيح'], 400);
        }

        return redirect()->to($redirectUrl);
    }


public function tracking($id, Request $request)
{
    $trackingLink = TrackingLink::find($id);

    if (!$trackingLink) {
        Log::error('رابط التتبع غير موجود', ['id' => $id]);
        return response()->json(['message' => 'رابط التتبع غير موجود'], 404);
    }

    // تحديث عدد الزيارات إذا كانت موجودة في الطلب
    if ($request->has('visits')) {
        $visits = $request->input('visits', 0);
        $trackingLink->increment('visits', $visits);
        Log::info('تم تحديث عدد الزيارات', ['tracking_link_id' => $id, 'visits' => $trackingLink->visits]);
    }

    // تحديث عدد عمليات الشراء إذا كانت موجودة في الطلب
    if ($request->has('purchases_count')) {
        $purchasesCount = $request->input('purchases_count', 0);
        $trackingLink->increment('purchases_count', $purchasesCount);
        Log::info('تم تحديث عدد عمليات الشراء', ['tracking_link_id' => $id, 'purchases_count' => $trackingLink->purchases_count]);
    }

    // تحديث معلومات التبرعات إذا كانت موجودة في الطلب
    if ($request->has('donation_id')) {
        $donationId = $request->input('donation_id');
        $donationAmount = $request->input('donation_amount', 0);

        $trackingLink->increment('donations');
        $trackingLink->update([
            'last_donation_id' => $donationId,
            'last_donation_amount' => $donationAmount,
        ]);

        Log::info('تم تحديث تفاصيل التبرع', [
            'tracking_link_id' => $id,
            'donation_id' => $donationId,
            'donation_amount' => $donationAmount,
            'donations' => $trackingLink->donations,
        ]);
    }

    return response()->json([
        'message' => 'تم تحديث البيانات بنجاح',
        'data' => $trackingLink,
    ], 200);
}



    public function show($id)
    {
        $trackingLink = TrackingLink::find($id);

        if (!$trackingLink) {
            return response()->json(['message' => 'رابط التتبع غير موجود'], 404);
        }

        return response()->json([
            'id' => $trackingLink->id,
            'title' => $trackingLink->title,
            'advertiser_name' => $trackingLink->advertiser_name,
            'advertiser_phone' => $trackingLink->advertiser_phone,
            'advertiser_email' => $trackingLink->advertiser_email,
            'custom_keyword' => $trackingLink->custom_keyword,
            'added_date' => $trackingLink->added_date,
            'visits' => $trackingLink->visits,
            'new_donors' => $trackingLink->new_donors,
            'donation_transactions' => $trackingLink->donation_transactions,
            'donations' => $trackingLink->donations,
            'url' => $trackingLink->url,
            'purchases_count' => $trackingLink->purchases_count,
            'is_archived' => $trackingLink->is_archived,
            'created_at' => $trackingLink->created_at->toDateTimeString(),
            'updated_at' => $trackingLink->updated_at->toDateTimeString(),
            'coordinator_id' => $trackingLink->coordinator_id,
        ], 200);
    }

    public function index()
    {
        $trackingLinks = TrackingLink::all();

        $trackingLinks->transform(function ($link) {
            $link->url = url("api/tracking/{$link->id}");
            return $link;
        });

        Log::info('تم جلب جميع الروابط التتبعية');

        return response()->json($trackingLinks, 200);
    }

    public function archive($id)
    {
        $trackingLink = TrackingLink::findOrFail($id);
        $trackingLink->is_archived = true;
        $trackingLink->save();

        Log::info('تم أرشفة الرابط بنجاح', ['tracking_link_id' => $id]);

        return response()->json(['message' => 'تمت أرشفة الرابط بنجاح']);
    }
    
    public function unarchive($id)
{
    $trackingLink = TrackingLink::findOrFail($id);
    $trackingLink->is_archived = false; // إلغاء الأرشفة
    $trackingLink->save();

    Log::info('تم إلغاء أرشفة الرابط بنجاح', ['tracking_link_id' => $id]);

    return response()->json(['message' => 'تم إلغاء أرشفة الرابط بنجاح']);
}

}
